import {
  useEditor,
  useValue
} from "@tldraw/editor";
import { getArrowBindings } from "../../shapes/arrow/shared.mjs";
function shapesWithUnboundArrows(editor) {
  const selectedShapeIds = editor.getSelectedShapeIds();
  const selectedShapes = selectedShapeIds.map((id) => {
    return editor.getShape(id);
  });
  return selectedShapes.filter((shape) => {
    if (!shape) return false;
    if (editor.isShapeOfType(shape, "arrow")) {
      const bindings = getArrowBindings(editor, shape);
      if (bindings.start || bindings.end) return false;
    }
    return true;
  });
}
const useThreeStackableItems = () => {
  const editor = useEditor();
  return useValue("threeStackableItems", () => shapesWithUnboundArrows(editor).length > 2, [editor]);
};
const useIsInSelectState = () => {
  const editor = useEditor();
  return useValue("isInSelectState", () => editor.isIn("select"), [editor]);
};
const useAllowGroup = () => {
  const editor = useEditor();
  return useValue(
    "allow group",
    () => {
      const selectedShapes = editor.getSelectedShapes();
      if (selectedShapes.length < 2) return false;
      for (const shape of selectedShapes) {
        if (editor.isShapeOfType(shape, "arrow")) {
          const bindings = getArrowBindings(editor, shape);
          if (bindings.start) {
            if (!selectedShapes.some((s) => s.id === bindings.start.toId)) {
              return false;
            }
          }
          if (bindings.end) {
            if (!selectedShapes.some((s) => s.id === bindings.end.toId)) {
              return false;
            }
          }
        }
      }
      return true;
    },
    [editor]
  );
};
const useAllowUngroup = () => {
  const editor = useEditor();
  return useValue(
    "allowUngroup",
    () => editor.getSelectedShapeIds().some((id) => editor.getShape(id)?.type === "group"),
    [editor]
  );
};
const showMenuPaste = typeof window !== "undefined" && "navigator" in window && Boolean(navigator.clipboard) && Boolean(navigator.clipboard.read);
function useAnySelectedShapesCount(min, max) {
  const editor = useEditor();
  return useValue(
    "selectedShapes",
    () => {
      const len = editor.getSelectedShapes().length;
      if (min === void 0) {
        if (max === void 0) {
          return len;
        } else {
          return len <= max;
        }
      } else {
        if (max === void 0) {
          return len >= min;
        } else {
          return len >= min && len <= max;
        }
      }
    },
    [editor, min, max]
  );
}
function useUnlockedSelectedShapesCount(min, max) {
  const editor = useEditor();
  return useValue(
    "selectedShapes",
    () => {
      const len = editor.getSelectedShapes().filter((s) => !editor.isShapeOrAncestorLocked(s)).length;
      if (min === void 0) {
        if (max === void 0) {
          return len;
        } else {
          return len <= max;
        }
      } else {
        if (max === void 0) {
          return len >= min;
        } else {
          return len >= min && len <= max;
        }
      }
    },
    [editor]
  );
}
function useShowAutoSizeToggle() {
  const editor = useEditor();
  return useValue(
    "showAutoSizeToggle",
    () => {
      const selectedShapes = editor.getSelectedShapes();
      return selectedShapes.length === 1 && editor.isShapeOfType(selectedShapes[0], "text") && selectedShapes[0].props.autoSize === false;
    },
    [editor]
  );
}
function useHasLinkShapeSelected() {
  const editor = useEditor();
  return useValue(
    "hasLinkShapeSelected",
    () => {
      const onlySelectedShape = editor.getOnlySelectedShape();
      return !!(onlySelectedShape && onlySelectedShape.type !== "embed" && "url" in onlySelectedShape.props && !onlySelectedShape.isLocked);
    },
    [editor]
  );
}
function useOnlyFlippableShape() {
  const editor = useEditor();
  return useValue(
    "onlyFlippableShape",
    () => {
      const shape = editor.getOnlySelectedShape();
      return shape && (editor.isShapeOfType(shape, "group") || editor.isShapeOfType(shape, "image") || editor.isShapeOfType(shape, "arrow") || editor.isShapeOfType(shape, "line") || editor.isShapeOfType(shape, "draw"));
    },
    [editor]
  );
}
function useCanRedo() {
  const editor = useEditor();
  return useValue("useCanRedo", () => editor.getCanRedo(), [editor]);
}
function useCanUndo() {
  const editor = useEditor();
  return useValue("useCanUndo", () => editor.getCanUndo(), [editor]);
}
export {
  showMenuPaste,
  useAllowGroup,
  useAllowUngroup,
  useAnySelectedShapesCount,
  useCanRedo,
  useCanUndo,
  useHasLinkShapeSelected,
  useIsInSelectState,
  useOnlyFlippableShape,
  useShowAutoSizeToggle,
  useThreeStackableItems,
  useUnlockedSelectedShapesCount
};
//# sourceMappingURL=menu-hooks.mjs.map
